<?php
class Lamkadeh_Filter_Product_Property_Widget extends WP_Widget
{

    public function __construct()
    {
        parent::__construct(
            'lamkadeh_filter_product_property',
            __('فیلتر ویژگی محصولات - لمکده', 'lamkadeh'),
            array('description' => __('نمایش دکمه فیلتر ویژگی در سایدبار فروشگاه', 'lamkadeh'))
        );

        // اضافه کردن فیلتر برای کوئری محصولات - با اولویت بالاتر
        add_action('woocommerce_product_query', array($this, 'filter_products_by_property'), 999);
    }

    // تابع فیلتر کردن محصولات بر اساس ویژگی
    public function filter_products_by_property($q)
    {
        // فقط در صفحات آرشیو محصولات فیلتر اعمال شود
        if (!is_product_category() && !is_shop() && !is_product_tag() && !is_product_taxonomy()) {
            return;
        }

        $selected_properties = $this->get_selected_properties_from_url();

        if (!empty($selected_properties)) {
            $tax_query = $q->get('tax_query');

            if (!is_array($tax_query)) {
                $tax_query = array();
            }

            // اضافه کردن فیلتر ویژگی به tax_query
            foreach ($selected_properties as $taxonomy => $terms) {
                if (!empty($terms)) {
                    // بررسی وجود taxonomy
                    if (!taxonomy_exists($taxonomy)) {
                        continue;
                    }

                    $tax_query[] = array(
                        'taxonomy' => $taxonomy,
                        'field'    => 'slug',
                        'terms'    => $terms,
                        'operator' => 'IN'
                    );
                }
            }

            // اگر بیش از یک فیلتر داریم، رابطه AND قرار می‌دهیم
            if (count($tax_query) > 1) {
                $tax_query['relation'] = 'AND';
            }

            $q->set('tax_query', $tax_query);

            // دیباگ برای بررسی کوئری
            error_log('=== LAMKADEH PROPERTY FILTER DEBUG ===');
            error_log('Selected Properties: ' . print_r($selected_properties, true));
            error_log('Final Tax Query: ' . print_r($tax_query, true));
            error_log('=== END DEBUG ===');
        }
    }

    // تابع دریافت ویژگی های انتخاب شده از URL
    private function get_selected_properties_from_url()
    {
        $selected_properties = array();

        // دریافت تمام پارامترهای GET
        foreach ($_GET as $key => $value) {
            // اگر پارامتر با pa_ شروع شود (ویژگی محصول ووکامرس)
            if (strpos($key, 'pa_') === 0 && taxonomy_exists($key)) {
                if (is_array($value)) {
                    $selected_properties[$key] = array_map('sanitize_text_field', $value);
                } else {
                    // استفاده مستقیم از مقدار فارسی
                    $selected_properties[$key] = array_map('trim', explode(',', sanitize_text_field($value)));
                }

                // حذف مقادیر خالی
                $selected_properties[$key] = array_filter($selected_properties[$key]);

                // دیباگ
                error_log("Property Filter - Taxonomy: $key, Terms: " . print_r($selected_properties[$key], true));
            }
        }

        return array_filter($selected_properties);
    }

    // تابع دریافت ID دسته‌بندی فعلی
    private function get_current_category_id()
    {
        if (is_product_category()) {
            $category = get_queried_object();
            return $category->term_id;
        }
        return 0;
    }

    // تابع دریافت ویژگی ها از ووکامرس با فیلتر دسته‌بندی
    private function get_product_properties($taxonomy, $category_id = 0, $hide_empty = true)
    {
        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => $hide_empty,
        );

        // اگر دسته‌بندی مشخص شده، کوئری محصولات رو فیلتر کن
        if ($category_id > 0) {
            // اول همه ویژگی ها رو بگیر
            $all_properties = get_terms($args);

            if (is_wp_error($all_properties) || empty($all_properties)) {
                return array();
            }

            // حالا برای هر ویژگی بررسی کن که در این دسته‌بندی محصول داره یا نه
            $properties = array();
            foreach ($all_properties as $term) {
                // کوئری برای محصولاتی که هم در این ویژگی و هم در این دسته‌بندی هستن
                $products_count = $this->get_products_count_by_property_and_category($term->term_id, $taxonomy, $category_id);

                // اگر hide_empty فعال هست و محصولی نداره، ردش کن
                if ($hide_empty && $products_count === 0) {
                    continue;
                }

                $properties[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'count' => $products_count,
                    'url' => get_term_link($term)
                );
            }
        } else {
            // اگر دسته‌بندی مشخص نشده، همه ویژگی ها رو بگیر
            $terms = get_terms($args);

            if (is_wp_error($terms)) {
                return array();
            }

            $properties = array();
            foreach ($terms as $term) {
                $properties[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'count' => $term->count,
                    'url' => get_term_link($term)
                );
            }
        }

        // مرتب‌سازی بر اساس نام
        usort($properties, function ($a, $b) {
            return strcmp($a['name'], $b['name']);
        });

        return $properties;
    }

    // تابع شمارش محصولات در یک ویژگی و دسته‌بندی خاص
    private function get_products_count_by_property_and_category($property_term_id, $taxonomy, $category_id)
    {
        $args = array(
            'status' => 'publish',
            'limit' => -1,
            'return' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $property_term_id,
                ),
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category_id,
                )
            )
        );

        $products = wc_get_products($args);
        return count($products);
    }

    // تابع رندر لیست ویژگی ها
    private function render_properties_list($properties, $selected_properties = array(), $taxonomy = '')
    {
        if (empty($properties)) {
            echo '<div class="no-properties-message">' . __('ویژگی در این دسته‌بندی یافت نشد.', 'lamkadeh') . '</div>';
            return;
        }

        foreach ($properties as $property) {
            $is_active = false;
            if (isset($selected_properties[$taxonomy])) {
                $is_active = in_array($property['slug'], $selected_properties[$taxonomy]);
            }
?>
            <div class="content-property-item <?php echo $is_active ? 'active' : ''; ?>"
                data-property-slug="<?php echo esc_attr($property['slug']); ?>"
                data-property-name="<?php echo esc_attr($property['name']); ?>"
                data-taxonomy="<?php echo esc_attr($taxonomy); ?>">
                <div class="content-property-item-data">
                    <span><?php echo esc_html($property['name']); ?></span>
                    <?php if ($property['count'] > 0) : ?>
                        <span><?php echo $property['count'] . ' ' . __('کالا', 'lamkadeh'); ?></span>
                    <?php endif; ?>
                </div>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <circle cx="12" cy="12" r="10" stroke="white" stroke-width="1.5" />
                    <path d="M8.5 12.5L10.5 14.5L15.5 9.5" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" />
                </svg>
            </div>
        <?php
        }
    }

    public function widget($args, $instance)
    {
        // فقط در صفحات مربوطه نمایش داده شود
        if (!is_product_category() && !is_shop() && !is_product_tag() && !is_product_taxonomy()) {
            return;
        }

        // دریافت تنظیمات
        $title = !empty($instance['title']) ? $instance['title'] : __('ویژگی محصولات', 'lamkadeh');
        $attribute_name = !empty($instance['attribute']) ? $instance['attribute'] : '';
        $hide_empty = !empty($instance['hide_empty']) ? (bool)$instance['hide_empty'] : true;

        if (empty($attribute_name)) {
            return;
        }

        $taxonomy = 'pa_' . $attribute_name;

        // بررسی وجود taxonomy
        if (!taxonomy_exists($taxonomy)) {
            echo '<p>خطا: ویژگی "' . esc_html($attribute_name) . '" وجود ندارد.</p>';
            return;
        }

        // دریافت ویژگی های انتخاب شده از URL
        $selected_properties = $this->get_selected_properties_from_url();
        $current_category_id = $this->get_current_category_id();
        $all_properties = $this->get_product_properties($taxonomy, $current_category_id, $hide_empty);

        // ایجاد لیست ویژگی ها برای JavaScript
        $properties_data = array();
        foreach ($all_properties as $property) {
            $properties_data[$property['slug']] = $property['name'];
        }

        // بررسی آیا ویژگی ای انتخاب شده یا نه
        $current_selected = isset($selected_properties[$taxonomy]) ? $selected_properties[$taxonomy] : array();
        $has_selected_properties = !empty($current_selected);

        echo $args['before_widget'];
        ?>

        <section class="filter-product-property">
            <div class="filter-product-property-buttom <?php echo $has_selected_properties ? 'active' : ''; ?>">
                <span><?php echo esc_html($title); ?></span>
                <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21.0813 15.96C20.8597 15.96 20.638 16.0417 20.463 16.2167L13.9997 22.68L7.53632 16.2167C7.19798 15.8783 6.63798 15.8783 6.29965 16.2167C5.96132 16.555 5.96132 17.115 6.29965 17.4533L13.3813 24.535C13.7197 24.8733 14.2796 24.8733 14.618 24.535L21.6997 17.4533C22.038 17.115 22.038 16.555 21.6997 16.2167C21.5363 16.0417 21.303 15.96 21.0813 15.96Z" fill="white" />
                    <path d="M14 3.20833C13.5217 3.20833 13.125 3.60499 13.125 4.08333L13.125 23.7183C13.125 24.1967 13.5217 24.5933 14 24.5933C14.4783 24.5933 14.875 24.1967 14.875 23.7183L14.875 4.08333C14.875 3.60499 14.4783 3.20833 14 3.20833Z" fill="white" />
                </svg>

                <div class="content">
                    <!-- بخش ویژگی های انتخاب شده -->
                    <div class="content-property-me" id="selected-properties-container-<?php echo esc_attr($attribute_name); ?>" <?php echo empty($current_selected) ? 'style="display:none;"' : ''; ?>>
                        <span><?php echo __("ویژگی های انتخاب شمـــــــــا", "lamkadeh") ?></span>
                        <div class="content-property-me-items" id="selected-properties-items-<?php echo esc_attr($attribute_name); ?>">
                            <?php
                            foreach ($current_selected as $property_slug):
                                // فقط ویژگی هایی رو نمایش بده که موجود هستن
                                if (isset($properties_data[$property_slug])):
                                    $property_term = get_term_by('slug', $property_slug, $taxonomy);
                                    if ($property_term):
                            ?>
                                        <div class="content-property-me-item" data-property-slug="<?php echo esc_attr($property_slug); ?>" data-taxonomy="<?php echo esc_attr($taxonomy); ?>">
                                            <span><?php echo esc_html($property_term->name); ?></span>
                                            <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                                <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                            </svg>
                                        </div>
                            <?php
                                    endif;
                                endif;
                            endforeach;
                            ?>
                        </div>
                    </div>

                    <span><?php echo esc_html($title); ?></span>

                    <!-- لیست ویژگی ها -->
                    <div class="content-property">
                        <?php $this->render_properties_list($all_properties, $selected_properties, $taxonomy); ?>
                    </div>
                </div>
            </div>
        </section>

        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const filterWidget = {
                    init() {
                        // مقداردهی اولیه از URL
                        this.selectedProperties = this.getSelectedPropertiesFromURL();
                        this.propertiesData = <?php echo json_encode($properties_data); ?>;
                        this.currentCategoryId = <?php echo $current_category_id; ?>;
                        this.attributeName = '<?php echo esc_js($attribute_name); ?>';
                        this.taxonomy = '<?php echo esc_js($taxonomy); ?>';

                        console.log('🔧 Filter Widget Initialized:', {
                            selectedProperties: this.selectedProperties,
                            propertiesData: this.propertiesData,
                            taxonomy: this.taxonomy
                        });

                        // اگر ویژگی ای انتخاب شده، آکوردین رو active کن
                        if (this.hasSelectedProperties()) {
                            this.openAccordion();
                        }

                        this.bindEvents();
                        this.updateSelectedPropertiesUI();
                        this.updatePropertyItemsActiveState();
                    },

                    // تابع بررسی آیا ویژگی انتخاب شده یا نه
                    hasSelectedProperties() {
                        return this.selectedProperties[this.taxonomy] && this.selectedProperties[this.taxonomy].length > 0;
                    },

                    // تابع باز کردن آکوردین
                    openAccordion() {
                        const accordionButton = document.querySelector('.filter-product-property-buttom');
                        if (accordionButton) {
                            accordionButton.classList.add('active');
                        }
                    },

                    // تابع بستن آکوردین
                    closeAccordion() {
                        const accordionButton = document.querySelector('.filter-product-property-buttom');
                        if (accordionButton) {
                            accordionButton.classList.remove('active');
                        }
                    },

                    // تابع دریافت ویژگی های انتخاب شده از URL
                    getSelectedPropertiesFromURL() {
                        const urlParams = new URLSearchParams(window.location.search);
                        const selectedProperties = {};

                        // دریافت تمام پارامترهایی که با pa_ شروع می‌شوند
                        urlParams.forEach((value, key) => {
                            if (key.startsWith('pa_')) {
                                // استفاده مستقیم از مقدار فارسی بدون decode
                                selectedProperties[key] = value.split(',').filter(slug => slug.trim() !== '');

                                console.log('📥 URL Parameter - Key:', key, 'Value:', value, 'Final array:', selectedProperties[key]);
                            }
                        });

                        return selectedProperties;
                    },

                    bindEvents() {
                        // آکاردیون
                        const accordionButton = document.querySelector('.filter-product-property-buttom');
                        if (!accordionButton) return;

                        const content = accordionButton.querySelector('.content');

                        accordionButton.addEventListener('click', function(e) {
                            if (content.contains(e.target)) {
                                return;
                            }
                            if (e.target.tagName.toLowerCase() === 'svg' ||
                                e.target.tagName.toLowerCase() === 'path' ||
                                e.target.tagName.toLowerCase() === 'span') {
                                this.classList.toggle('active');
                            }
                        });

                        // کلیک روی ویژگی
                        document.addEventListener('click', (e) => {
                            const propertyItem = e.target.closest('.content-property-item');
                            if (propertyItem) {
                                this.toggleProperty(propertyItem);
                                return;
                            }

                            // حذف ویژگی
                            const removeBtn = e.target.closest('.content-property-me-item');
                            if (removeBtn) {
                                this.removeProperty(removeBtn.dataset.propertySlug, removeBtn.dataset.taxonomy);
                            }
                        });
                    },

                    // تابع به روزرسانی وضعیت active آیتم‌ها
                    updatePropertyItemsActiveState() {
                        const propertyItems = document.querySelectorAll('.content-property-item');
                        const currentSelected = this.selectedProperties[this.taxonomy] || [];

                        console.log('🎯 Updating active state - Selected:', currentSelected);
                        console.log('🎯 Total property items found:', propertyItems.length);

                        propertyItems.forEach(item => {
                            const slug = item.dataset.propertySlug;
                            if (currentSelected.includes(slug)) {
                                item.classList.add('active');
                                console.log('✅ Adding active class to:', slug);
                            } else {
                                item.classList.remove('active');
                            }
                        });
                    },

                    toggleProperty(propertyItem) {
                        const propertySlug = propertyItem.dataset.propertySlug;
                        const taxonomy = propertyItem.dataset.taxonomy;

                        console.log('🔄 Toggling property:', propertySlug, taxonomy);

                        if (!this.selectedProperties[taxonomy]) {
                            this.selectedProperties[taxonomy] = [];
                        }

                        if (this.selectedProperties[taxonomy].includes(propertySlug)) {
                            // حذف ویژگی
                            this.selectedProperties[taxonomy] = this.selectedProperties[taxonomy].filter(slug => slug !== propertySlug);
                            propertyItem.classList.remove('active');
                            console.log('➖ Removed property:', propertySlug);
                        } else {
                            // اضافه کردن ویژگی
                            this.selectedProperties[taxonomy].push(propertySlug);
                            propertyItem.classList.add('active');
                            console.log('➕ Added property:', propertySlug);
                        }

                        console.log('📊 After toggle:', this.selectedProperties);

                        this.updateSelectedPropertiesUI();

                        // اگر ویژگی ای انتخاب شده، آکوردین رو باز کن
                        if (this.hasSelectedProperties()) {
                            this.openAccordion();
                        } else {
                            this.closeAccordion();
                        }

                        this.updateURL();
                    },

                    removeProperty(propertySlug, taxonomy) {
                        console.log('🗑️ Removing property:', propertySlug, taxonomy);

                        if (this.selectedProperties[taxonomy]) {
                            this.selectedProperties[taxonomy] = this.selectedProperties[taxonomy].filter(slug => slug !== propertySlug);
                        }

                        // به‌روزرسانی وضعیت آیتم در لیست
                        this.updatePropertyItemsActiveState();

                        this.updateSelectedPropertiesUI();

                        // اگر هیچ ویژگی ای انتخاب نشده، آکوردین رو ببند
                        if (!this.hasSelectedProperties()) {
                            this.closeAccordion();
                        }

                        this.updateURL();
                    },

                    updateSelectedPropertiesUI() {
                        const container = document.getElementById('selected-properties-container-<?php echo esc_js($attribute_name); ?>');
                        const itemsContainer = document.getElementById('selected-properties-items-<?php echo esc_js($attribute_name); ?>');

                        if (!container || !itemsContainer) return;

                        itemsContainer.innerHTML = '';

                        const currentSelected = this.selectedProperties[this.taxonomy] || [];

                        // فقط ویژگی هایی رو نمایش بده که در propertiesData موجود هستن
                        const validSelectedProperties = currentSelected.filter(slug => this.propertiesData[slug]);

                        console.log('📝 Updating UI - Valid selected:', validSelectedProperties);

                        if (validSelectedProperties.length === 0) {
                            container.style.display = 'none';
                            console.log('📭 No selected properties to show');
                            return;
                        }

                        container.style.display = 'block';
                        console.log('📬 Showing selected properties container');

                        validSelectedProperties.forEach(slug => {
                            const propertyName = this.propertiesData[slug];

                            if (propertyName) {
                                const item = document.createElement('div');
                                item.className = 'content-property-me-item';
                                item.dataset.propertySlug = slug;
                                item.dataset.taxonomy = this.taxonomy;
                                item.innerHTML = `
                                <span>${propertyName}</span>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                </svg>
                            `;
                                itemsContainer.appendChild(item);
                                console.log('📋 Added selected property to UI:', propertyName);
                            }
                        });
                    },

                    updateURL() {
                        const url = new URL(window.location.href);

                        // حذف تمام پارامترهای ویژگی های قبلی
                        const paramsToDelete = [];
                        url.searchParams.forEach((value, key) => {
                            if (key.startsWith('pa_')) {
                                paramsToDelete.push(key);
                            }
                        });

                        paramsToDelete.forEach(key => {
                            url.searchParams.delete(key);
                        });

                        // اضافه کردن ویژگی های جدید - به صورت فارسی
                        Object.keys(this.selectedProperties).forEach(taxonomy => {
                            const selected = this.selectedProperties[taxonomy].filter(slug => this.propertiesData[slug]);

                            if (selected.length > 0) {
                                // استفاده مستقیم از مقادیر فارسی بدون encode
                                const value = selected.join(',');
                                url.searchParams.set(taxonomy, value);
                                console.log('🔗 Setting URL parameter:', taxonomy, '=', value);
                            }
                        });

                        console.log('🌐 Updating URL to:', url.toString());

                        // رفرش صفحه با URL جدید
                        window.location.href = url.toString();
                    }
                };

                filterWidget.init();
            });
        </script>

    <?php
        echo $args['after_widget'];
    }

    public function form($instance)
    {
        // مقداردهی پیش‌فرض
        $title = !empty($instance['title']) ? $instance['title'] : __('ویژگی محصولات', 'lamkadeh');
        $attribute = !empty($instance['attribute']) ? $instance['attribute'] : '';
        $hide_empty = isset($instance['hide_empty']) ? (bool)$instance['hide_empty'] : true;

        // دریافت تمام ویژگی های محصول
        $attribute_taxonomies = wc_get_attribute_taxonomies();
    ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('عنوان:', 'lamkadeh'); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>"
                name="<?php echo $this->get_field_name('title'); ?>" type="text"
                value="<?php echo esc_attr($title); ?>">
        </p>

        <p>
            <label for="<?php echo $this->get_field_id('attribute'); ?>"><?php _e('ویژگی:', 'lamkadeh'); ?></label>
            <select class="widefat" id="<?php echo $this->get_field_id('attribute'); ?>"
                name="<?php echo $this->get_field_name('attribute'); ?>">
                <option value=""><?php _e('-- انتخاب ویژگی --', 'lamkadeh'); ?></option>
                <?php foreach ($attribute_taxonomies as $tax) : ?>
                    <option value="<?php echo esc_attr($tax->attribute_name); ?>" <?php selected($attribute, $tax->attribute_name); ?>>
                        <?php echo esc_html($tax->attribute_label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </p>

        <p>
            <input type="checkbox"
                id="<?php echo $this->get_field_id('hide_empty'); ?>"
                name="<?php echo $this->get_field_name('hide_empty'); ?>"
                value="1"
                <?php checked($hide_empty, true); ?> />
            <label for="<?php echo $this->get_field_id('hide_empty'); ?>">
                <?php echo __('مخفی کردن ویژگی های خالی', 'lamkadeh'); ?>
            </label>
            <br>
            <small><?php echo __('اگر فعال باشد، ویژگی هایی که محصولی ندارند نمایش داده نمی‌شوند.', 'lamkadeh'); ?></small>
        </p>
<?php
    }

    public function update($new_instance, $old_instance)
    {
        $instance = array();
        $instance['title'] = !empty($new_instance['title']) ? strip_tags($new_instance['title']) : '';
        $instance['attribute'] = !empty($new_instance['attribute']) ? strip_tags($new_instance['attribute']) : '';
        $instance['hide_empty'] = !empty($new_instance['hide_empty']) ? 1 : 0;
        return $instance;
    }
}

add_action('widgets_init', function () {
    register_widget('Lamkadeh_Filter_Product_Property_Widget');
});
