<?php
class Lamkadeh_Filter_Product_Price_Widget extends WP_Widget
{
    function __construct()
    {
        parent::__construct(
            'lamkadeh_filter_product_price',
            __('فیلتر قیمت محصولات - لمکده', 'lamkadeh'),
            array('description' => __('فیلتر های قیمت محصولات', 'lamkadeh'))
        );

        // اضافه کردن فیلتر برای کوئری محصولات
        add_action('woocommerce_product_query', array($this, 'filter_products_by_price'));
    }

    // تابع فیلتر کردن محصولات بر اساس قیمت
    public function filter_products_by_price($q)
    {
        $min_price = $this->get_min_price_from_url();
        $max_price = $this->get_max_price_from_url();

        if ($min_price !== '' || $max_price !== '') {
            $meta_query = $q->get('meta_query');

            if (!is_array($meta_query)) {
                $meta_query = array();
            }

            $price_query = array();

            if ($min_price !== '') {
                $price_query[] = array(
                    'key' => '_price',
                    'value' => floatval($min_price),
                    'compare' => '>=',
                    'type' => 'NUMERIC'
                );
            }

            if ($max_price !== '') {
                $price_query[] = array(
                    'key' => '_price',
                    'value' => floatval($max_price),
                    'compare' => '<=',
                    'type' => 'NUMERIC'
                );
            }

            if (!empty($price_query)) {
                $price_query['relation'] = 'AND';
                $meta_query[] = $price_query;
                $q->set('meta_query', $meta_query);
            }
        }
    }

    // تابع دریافت حداقل قیمت از URL
    private function get_min_price_from_url()
    {
        return isset($_GET['min_price']) ? sanitize_text_field($_GET['min_price']) : '';
    }

    // تابع دریافت حداکثر قیمت از URL
    private function get_max_price_from_url()
    {
        return isset($_GET['max_price']) ? sanitize_text_field($_GET['max_price']) : '';
    }

    // تابع دریافت محدوده قیمت محصولات در آرشیو فعلی
    private function get_price_range()
    {
        global $wpdb;

        // دریافت محصولات آرشیو فعلی با کوئری مستقیم
        $product_ids = $this->get_current_archive_product_ids();

        if (empty($product_ids)) {
            return array(
                'min' => 0,
                'max' => 1000000
            );
        }

        $product_ids_str = implode(',', array_map('intval', $product_ids));

        // دریافت حداقل و حداکثر قیمت از محصولات این آرشیو
        $price_range = $wpdb->get_row("
            SELECT 
                MIN(CAST(meta_value AS DECIMAL)) as min_price,
                MAX(CAST(meta_value AS DECIMAL)) as max_price
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_price' 
            AND post_id IN ({$product_ids_str})
        ");

        // دیباگ
        error_log("Price Range Debug - Found " . count($product_ids) . " products in current archive");
        if ($price_range) {
            error_log("Price Range Debug - Min: " . ($price_range->min_price ?? 'null') . ", Max: " . ($price_range->max_price ?? 'null'));
        } else {
            error_log("Price Range Debug - No price data found");
        }

        if ($price_range && $price_range->min_price !== null && $price_range->max_price !== null) {
            return array(
                'min' => floatval($price_range->min_price),
                'max' => floatval($price_range->max_price)
            );
        }

        // مقادیر پیش‌فرض اگر محصولی وجود نداشت
        return array(
            'min' => 0,
            'max' => 1000000
        );
    }

    // تابع دریافت ID محصولات آرشیو فعلی
    private function get_current_archive_product_ids()
    {
        global $wp_query;

        // اگر در آرشیو ووکامرس هستیم و محصولاتی وجود دارند
        if (is_woocommerce() && $wp_query->posts) {
            $product_ids = array();

            // جمع‌آوری ID محصولات از کوئری اصلی
            foreach ($wp_query->posts as $post) {
                if ($post->post_type === 'product') {
                    $product_ids[] = $post->ID;
                }
            }

            // اگر محصولی پیدا شد برگردان
            if (!empty($product_ids)) {
                error_log("Found " . count($product_ids) . " products in main query");
                return $product_ids;
            }
        }

        // روش جایگزین: ایجاد کوئری بر اساس آرشیو فعلی
        return $this->get_archive_product_ids_by_query();
    }

    // تابع جایگزین برای دریافت ID محصولات
    private function get_archive_product_ids_by_query()
    {
        global $wp_query;

        // ایجاد کوئری بر اساس آرشیو فعلی
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => 500, // محدودیت معقول
            'fields' => 'ids',
            'no_found_rows' => true,
            'update_post_term_cache' => false,
            'update_post_meta_cache' => false
        );

        // اگر در دسته‌بندی محصول هستیم
        if (is_product_category() && get_queried_object()) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'id',
                    'terms' => get_queried_object_id()
                )
            );
        }
        // اگر در تگ محصول هستیم
        elseif (is_product_tag() && get_queried_object()) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'product_tag',
                    'field' => 'id',
                    'terms' => get_queried_object_id()
                )
            );
        }
        // اگر در جستجو هستیم
        elseif (is_search()) {
            $args['s'] = get_search_query();
        }
        // اگر در فروشگاه هستیم، از کوئری اصلی استفاده کن
        elseif (is_shop() && !empty($wp_query->query_vars)) {
            $args = $wp_query->query_vars;
            $args['posts_per_page'] = 500;
            $args['fields'] = 'ids';
            $args['no_found_rows'] = true;
            $args['update_post_term_cache'] = false;
            $args['update_post_meta_cache'] = false;

            // حذف پارامترهای قیمت
            if (isset($args['meta_query'])) {
                foreach ($args['meta_query'] as $key => $meta) {
                    if (isset($meta['key']) && $meta['key'] === '_price') {
                        unset($args['meta_query'][$key]);
                    }
                }
                if (empty($args['meta_query'])) {
                    unset($args['meta_query']);
                }
            }
        }

        // اجرای کوئری
        $products_query = new WP_Query($args);
        $product_ids = $products_query->posts;

        error_log("Alternative query found " . count($product_ids) . " products");

        return $product_ids;
    }

    // تابع فرمت کردن قیمت
    private function format_price($price)
    {
        return number_format($price) . ' ' . __('تومان', 'lamkadeh');
    }

    function widget($args, $instance)
    {
        // دریافت محدوده قیمت
        $price_range = $this->get_price_range();
        $min_price = $price_range['min'];
        $max_price = $price_range['max'];

        // دیباگ نهایی
        error_log("Final Price Range - Min: $min_price, Max: $max_price");

        // اگر min و max برابر باشند یا min بیشتر از max باشد، مقادیر پیش‌فرض قرار بده
        if ($min_price >= $max_price || $min_price == 0 && $max_price == 1000000) {
            // بررسی دستی - اگر محصولی در آرشیو هست اما قیمت‌ها درست نیستند
            $manual_range = $this->get_manual_price_range();
            if ($manual_range['min'] != 0 || $manual_range['max'] != 1000000) {
                $min_price = $manual_range['min'];
                $max_price = $manual_range['max'];
                error_log("Using manual price range - Min: $min_price, Max: $max_price");
            }
        }

        // دریافت قیمت‌های انتخاب شده از URL
        $selected_min_price = $this->get_min_price_from_url();
        $selected_max_price = $this->get_max_price_from_url();

        // بررسی آیا قیمتی انتخاب شده یا نه
        $has_selected_price = !empty($selected_min_price) || !empty($selected_max_price);

        // محاسبه درصدهای اولیه
        $left_percent = 0;
        $right_percent = 1;

        if ($selected_min_price !== '' && $max_price > $min_price) {
            $left_percent = ($selected_min_price - $min_price) / ($max_price - $min_price);
        }

        if ($selected_max_price !== '' && $max_price > $min_price) {
            $right_percent = ($selected_max_price - $min_price) / ($max_price - $min_price);
        }

        // محدود کردن درصدها به بازه 0-1
        $left_percent = max(0, min(1, $left_percent));
        $right_percent = max(0, min(1, $right_percent));

        // اطمینان از اینکه left همیشه کوچکتر از right باشد
        if ($left_percent >= $right_percent) {
            $left_percent = 0;
            $right_percent = 1;
        }

        // محاسبه مقادیر فعلی برای نمایش
        $current_min_display = $selected_min_price !== '' ? $selected_min_price : $min_price;
        $current_max_display = $selected_max_price !== '' ? $selected_max_price : $max_price;

        echo $args['before_widget'];
?>
        <section class="filter-product-price">
            <div class="filter-product-price-buttom <?php echo $has_selected_price ? 'active' : ''; ?>">
                <span><?php echo __("محدوده قیمت", "lamkadeh"); ?></span>
                <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21.0813 15.96C20.8597 15.96 20.638 16.0417 20.463 16.2167L13.9997 22.68L7.53632 16.2167C7.19798 15.8783 6.63798 15.8783 6.29965 16.2167C5.96132 16.555 5.96132 17.115 6.29965 17.4533L13.3813 24.535C13.7197 24.8733 14.2796 24.8733 14.618 24.535L21.6997 17.4533C22.038 17.115 22.038 16.555 21.6997 16.2167C21.5363 16.0417 21.303 15.96 21.0813 15.96Z" fill="white" />
                    <path d="M14 3.20833C13.5217 3.20833 13.125 3.60499 13.125 4.08333L13.125 23.7183C13.125 24.1967 13.5217 24.5933 14 24.5933C14.4783 24.5933 14.875 24.1967 14.875 23.7183L14.875 4.08333C14.875 3.60499 14.4783 3.20833 14 3.20833Z" fill="white" />
                </svg>
                <div class="content">
                    <div class="filter-product-price-box">
                        <span><?php echo __("از", "lamkadeh") ?></span>
                        <div class="filter-product-price-box-min-amount" id="min-price-display"><?php echo number_format($current_min_display); ?></div>
                        <span><?php echo __("تومان", "lamkadeh") ?></span>
                    </div>
                    <div class="filter-product-price-box">
                        <span><?php echo __("تا", "lamkadeh") ?></span>
                        <div class="filter-product-price-box-min-amount" id="max-price-display"><?php echo number_format($current_max_display); ?></div>
                        <span><?php echo __("تومان", "lamkadeh") ?></span>
                    </div>
                    <div class="filter-product-price-sider">
                        <div class="range-box">
                            <div class="range-values">
                                <span id="max-label"><?php echo number_format($max_price); ?></span>
                                <span id="min-label"><?php echo number_format($min_price); ?></span>
                            </div>
                            <div class="range-track" id="track">
                                <div class="range-fill" id="fill"></div>
                                <div class="knob" id="left-knob"></div>
                                <div class="knob" id="right-knob"></div>
                            </div>
                            <div class="avg-value" id="avg-box"><?php echo number_format(($min_price + $max_price) / 2); ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </section>

        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const priceFilter = {
                    init() {
                        this.minVal = <?php echo $min_price; ?>;
                        this.maxVal = <?php echo $max_price; ?>;
                        this.leftPercent = <?php echo $left_percent; ?>;
                        this.rightPercent = <?php echo $right_percent; ?>;

                        this.track = document.getElementById("track");
                        this.fill = document.getElementById("fill");
                        this.leftKnob = document.getElementById("left-knob");
                        this.rightKnob = document.getElementById("right-knob");
                        this.avgBox = document.getElementById("avg-box");
                        this.minPriceDisplay = document.getElementById("min-price-display");
                        this.maxPriceDisplay = document.getElementById("max-price-display");

                        this.activeKnob = null;

                        this.bindEvents();
                        this.updatePositions();
                        this.updatePriceDisplays();

                        // اگر قیمتی انتخاب شده، آکوردین رو active کن
                        if (this.hasSelectedPrice()) {
                            this.openAccordion();
                        }
                    },

                    // تابع باز کردن آکوردین
                    openAccordion() {
                        const accordionButton = document.querySelector('.filter-product-price-buttom');
                        accordionButton.classList.add('active');
                    },

                    // تابع بستن آکوردین
                    closeAccordion() {
                        const accordionButton = document.querySelector('.filter-product-price-buttom');
                        accordionButton.classList.remove('active');
                    },

                    // بررسی آیا قیمتی انتخاب شده یا نه
                    hasSelectedPrice() {
                        return <?php echo $has_selected_price ? 'true' : 'false'; ?>;
                    },

                    formatPrice(num) {
                        return new Intl.NumberFormat('fa-IR').format(num);
                    },

                    updatePositions() {
                        const trackWidth = this.track.offsetWidth;
                        const leftPx = this.leftPercent * trackWidth;
                        const rightPx = this.rightPercent * trackWidth;

                        this.leftKnob.style.left = `${leftPx}px`;
                        this.rightKnob.style.left = `${rightPx}px`;

                        this.fill.style.left = `${leftPx}px`;
                        this.fill.style.width = `${rightPx - leftPx}px`;

                        // مقدار عددی
                        const leftValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.leftPercent);
                        const rightValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.rightPercent);
                        const avg = Math.round((leftValue + rightValue) / 2);

                        this.avgBox.textContent = this.formatPrice(avg);
                    },

                    // تابع جدید برای به‌روزرسانی نمایش قیمت‌ها
                    updatePriceDisplays() {
                        const leftValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.leftPercent);
                        const rightValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.rightPercent);

                        // فقط اگر کاربر دستکاری کرده، نمایش بده
                        if (this.leftPercent > 0 || this.rightPercent < 1) {
                            this.minPriceDisplay.textContent = this.formatPrice(leftValue);
                            this.maxPriceDisplay.textContent = this.formatPrice(rightValue);
                        } else {
                            this.minPriceDisplay.textContent = this.formatPrice(this.minVal);
                            this.maxPriceDisplay.textContent = this.formatPrice(this.maxVal);
                        }
                    },

                    handleDrag(e) {
                        if (!this.activeKnob) return;
                        const rect = this.track.getBoundingClientRect();
                        const posX = e.clientX - rect.left;
                        let percent = posX / rect.width;

                        if (percent < 0) percent = 0;
                        if (percent > 1) percent = 1;

                        if (this.activeKnob === this.leftKnob && percent < this.rightPercent - 0.02) {
                            this.leftPercent = percent;
                        } else if (this.activeKnob === this.rightKnob && percent > this.leftPercent + 0.02) {
                            this.rightPercent = percent;
                        }

                        this.updatePositions();
                        this.updatePriceDisplays();
                    },

                    updateURL() {
                        const leftValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.leftPercent);
                        const rightValue = Math.round(this.minVal + (this.maxVal - this.minVal) * this.rightPercent);

                        const url = new URL(window.location.href);

                        if (leftValue > this.minVal) {
                            url.searchParams.set('min_price', leftValue);
                        } else {
                            url.searchParams.delete('min_price');
                        }

                        if (rightValue < this.maxVal) {
                            url.searchParams.set('max_price', rightValue);
                        } else {
                            url.searchParams.delete('max_price');
                        }

                        // رفرش صفحه با URL جدید
                        window.location.href = url.toString();
                    },

                    bindEvents() {
                        // آکاردیون - فقط روی دکمه اصلی کار کنه
                        const accordionButton = document.querySelector('.filter-product-price-buttom');

                        accordionButton.addEventListener('click', (e) => {
                            // فقط اگر روی خود دکمه اصلی کلیک شده (نه روی محتوای داخلش)
                            if (e.target === accordionButton ||
                                e.target.tagName.toLowerCase() === 'span' ||
                                e.target.tagName.toLowerCase() === 'svg' ||
                                e.target.tagName.toLowerCase() === 'path') {

                                // اگر روی content کلیک شده، کاری نکن
                                const content = accordionButton.querySelector('.content');
                                if (content.contains(e.target)) {
                                    return;
                                }

                                accordionButton.classList.toggle('active');
                            }
                        });

                        // کشیدن ناب
                        this.leftKnob.addEventListener("mousedown", (e) => {
                            e.preventDefault();
                            this.activeKnob = this.leftKnob;
                        });

                        this.rightKnob.addEventListener("mousedown", (e) => {
                            e.preventDefault();
                            this.activeKnob = this.rightKnob;
                        });

                        document.addEventListener("mouseup", () => {
                            if (this.activeKnob) {
                                this.activeKnob = null;
                                this.updateURL();
                            }
                        });

                        document.addEventListener("mousemove", (e) => {
                            this.handleDrag(e);
                        });

                        window.addEventListener("resize", () => {
                            this.updatePositions();
                        });
                    }
                };

                priceFilter.init();
            });
        </script>
    <?php
        echo $args['after_widget'];
    }

    // تابع جایگزین برای محاسبه دستی محدوده قیمت
    private function get_manual_price_range()
    {
        global $wpdb;

        // دریافت محصولات آرشیو فعلی
        $product_ids = $this->get_current_archive_product_ids();

        if (empty($product_ids)) {
            return array('min' => 0, 'max' => 1000000);
        }

        $product_ids_str = implode(',', array_map('intval', $product_ids));

        // کوئری مستقیم برای دریافت قیمت‌ها
        $prices = $wpdb->get_col("
            SELECT CAST(meta_value AS DECIMAL) as price
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_price' 
            AND post_id IN ({$product_ids_str})
            ORDER BY price ASC
        ");

        if (empty($prices)) {
            return array('min' => 0, 'max' => 1000000);
        }

        return array(
            'min' => floatval(min($prices)),
            'max' => floatval(max($prices))
        );
    }

    function form($instance)
    {
    ?>
        <p><?php echo __('این ویجت فیلتر قیمت محصولات را نمایش می‌دهد.', 'lamkadeh'); ?></p>
<?php
    }

    function update($new_instance, $old_instance)
    {
        return $new_instance;
    }
}

add_action('widgets_init', function () {
    register_widget('Lamkadeh_Filter_Product_Price_Widget');
});

// اضافه کردن فیلتر برای صفحات آرشیو محصولات
add_filter('woocommerce_shortcode_products_query', 'lamkadeh_filter_shortcode_products_by_price', 10, 2);
function lamkadeh_filter_shortcode_products_by_price($query_args, $attributes)
{
    $min_price = isset($_GET['min_price']) ? sanitize_text_field($_GET['min_price']) : '';
    $max_price = isset($_GET['max_price']) ? sanitize_text_field($_GET['max_price']) : '';

    if ($min_price !== '' || $max_price !== '') {
        if (!isset($query_args['meta_query'])) {
            $query_args['meta_query'] = array();
        }

        $price_query = array();

        if ($min_price !== '') {
            $price_query[] = array(
                'key' => '_price',
                'value' => floatval($min_price),
                'compare' => '>=',
                'type' => 'NUMERIC'
            );
        }

        if ($max_price !== '') {
            $price_query[] = array(
                'key' => '_price',
                'value' => floatval($max_price),
                'compare' => '<=',
                'type' => 'NUMERIC'
            );
        }

        if (!empty($price_query)) {
            $price_query['relation'] = 'AND';
            $query_args['meta_query'][] = $price_query;
        }
    }

    return $query_args;
}
