<?php
class Lamkadeh_Filter_Product_Color_Widget extends WP_Widget
{
    public function __construct()
    {
        parent::__construct(
            'lamkadeh_filter_product_color',
            __('فیلتر رنگ محصولات - لمکده', 'lamkadeh'),
            array('description' => __('نمایش فیلتر رنگ محصولات در سایدبار فروشگاه', 'lamkadeh'))
        );

        // اضافه کردن فیلتر برای کوئری محصولات
        add_action('woocommerce_product_query', array($this, 'filter_products_by_color'));
    }

    // تابع فیلتر کردن محصولات بر اساس رنگ
    public function filter_products_by_color($q)
    {
        $selected_colors = $this->get_selected_colors_from_url();

        if (!empty($selected_colors)) {
            $tax_query = $q->get('tax_query');

            if (!is_array($tax_query)) {
                $tax_query = array();
            }

            // اضافه کردن فیلتر رنگ به tax_query
            $color_tax_query = array(
                array(
                    'taxonomy' => 'pa_color',
                    'field'    => 'slug',
                    'terms'    => $selected_colors,
                    'operator' => 'IN'
                )
            );

            // اگر tax_query وجود داره، با رابطه AND اضافه کن
            if (!empty($tax_query)) {
                // اگر قبلاً relation تنظیم شده، رنگ رو بهش اضافه کن
                if (isset($tax_query['relation'])) {
                    $tax_query[] = $color_tax_query[0];
                } else {
                    $tax_query = array(
                        'relation' => 'AND',
                        $tax_query,
                        $color_tax_query[0]
                    );
                }
            } else {
                $tax_query = $color_tax_query;
            }

            $q->set('tax_query', $tax_query);
        }
    }

    // تابع دریافت رنگ‌های انتخاب شده از URL
    private function get_selected_colors_from_url()
    {
        $selected_colors = array();

        // استفاده از پارامتر product_color به جای color
        if (isset($_GET['product_color'])) {
            if (is_array($_GET['product_color'])) {
                $selected_colors = array_map('sanitize_text_field', $_GET['product_color']);
            } else {
                $selected_colors = array_map('trim', explode(',', sanitize_text_field($_GET['product_color'])));
            }
        }

        return array_filter($selected_colors);
    }

    // تابع دریافت ID دسته‌بندی فعلی
    private function get_current_category_id()
    {
        if (is_product_category()) {
            $category = get_queried_object();
            return $category->term_id;
        }
        return 0;
    }

    // تابع دریافت رنگ‌های محصولات
    private function get_product_colors()
    {
        $current_category_id = $this->get_current_category_id();

        // ابتدا همه رنگ‌های فعال رو بگیر
        $args = array(
            'taxonomy' => 'pa_color',
            'hide_empty' => true,
            'orderby' => 'name',
            'order' => 'ASC'
        );

        $all_color_terms = get_terms($args);

        if (is_wp_error($all_color_terms) || empty($all_color_terms)) {
            return array();
        }

        $colors = array();

        // برای هر رنگ بررسی کن که در دسته‌بندی فعلی محصول داره یا نه
        foreach ($all_color_terms as $term) {
            $products_count = 0;

            // اگر در دسته‌بندی خاصی هستیم، تعداد محصولات این رنگ در این دسته‌بندی رو بگیر
            if ($current_category_id > 0) {
                $products_count = $this->get_products_count_by_color_and_category($term->term_id, $current_category_id);
            } else {
                // اگر در آرشیو اصلی هستیم، از count خود ترم استفاده کن
                $products_count = $term->count;
            }

            // اگر محصولی وجود داره، این رنگ رو اضافه کن
            if ($products_count > 0) {
                $color_code = $this->get_color_code($term->term_id);

                $colors[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'count' => $products_count,
                    'color_code' => $color_code
                );
            }
        }

        return $colors;
    }

    // تابع شمارش محصولات در یک رنگ و دسته‌بندی خاص
    private function get_products_count_by_color_and_category($color_term_id, $category_id)
    {
        $args = array(
            'status' => 'publish',
            'limit' => -1,
            'return' => 'ids',
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => 'pa_color',
                    'field' => 'term_id',
                    'terms' => $color_term_id,
                ),
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category_id,
                )
            )
        );

        $products = wc_get_products($args);
        return count($products);
    }

    // تابع دریافت کد رنگ از attribute
    private function get_color_code($term_id)
    {
        // دریافت مستقیم کد رنگ از term_meta
        $color_code = get_term_meta($term_id, 'attribute_color', true);

        // دیباگ: چک کن چه مقداری برگردونده میشه
        error_log("Color Code for term_id {$term_id}: " . $color_code);

        // اگر کد رنگ وجود داره و معتبره، برگردونش
        if (!empty($color_code) && $this->is_valid_hex_color($color_code)) {
            return $color_code;
        }

        // اگر کد رنگ وجود نداره یا معتبر نیست، از نام رنگ استفاده کن
        $term = get_term($term_id, 'pa_color');
        if ($term && !is_wp_error($term)) {
            $fallback_color = $this->get_color_code_from_name($term->name);
            error_log("Using fallback color for {$term->name}: " . $fallback_color);
            return $fallback_color;
        }

        return '#CCCCCC';
    }

    // تابع بررسی معتبر بودن کد HEX رنگ
    private function is_valid_hex_color($color)
    {
        return preg_match('/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $color);
    }

    // تابع تبدیل نام رنگ به کد HEX
    private function get_color_code_from_name($color_name)
    {
        $color_map = array(
            'سبز' => '#008000',
            'قرمز' => '#FF0000',
            'آبی' => '#0000FF',
            'زرد' => '#FFFF00',
            'مشکی' => '#000000',
            'سفید' => '#FFFFFF',
            'نارنجی' => '#FFA500',
            'بنفش' => '#800080',
            'قهوه ای' => '#A52A2A',
            'قهوه‌ای' => '#A52A2A',
            'صورتی' => '#FFC0CB',
            'خاکستری' => '#808080',
            'نقره ای' => '#C0C0C0',
            'نقره‌ای' => '#C0C0C0',
            'طلایی' => '#FFD700',
            'نوک مدادی' => '#2F4F4F',
            'قهوه ای روشن' => '#A0522D',
            'قهوه‌ای روشن' => '#A0522D',
            'آبی تیره' => '#00008B',
            'سبز تیره' => '#006400',
            'سبز روشن' => '#90EE90',
            'آبی روشن' => '#ADD8E6',
            'بژ' => '#F5F5DC',
            'طلائی' => '#FFD700',
            'نقره‌ای' => '#C0C0C0',
        );

        $color_name = trim($color_name);
        return isset($color_map[$color_name]) ? $color_map[$color_name] : '#CCCCCC';
    }

    // تابع رندر لیست رنگ‌ها
    private function render_colors_list($colors, $selected_colors = array())
    {
        if (empty($colors)) {
            return;
        }

        foreach ($colors as $color) {
            $is_active = in_array($color['slug'], $selected_colors);
?>
            <div class="content-color-item <?php echo $is_active ? 'active' : ''; ?>"
                data-color-slug="<?php echo esc_attr($color['slug']); ?>"
                data-color-name="<?php echo esc_attr($color['name']); ?>">
                <div class="content-color-item-data">
                    <div class="content-radio-color-pa">
                        <div class="content-radio-color-pa-code first" style="background-color: <?php echo esc_attr($color['color_code']); ?>;"></div>
                        <div class="content-radio-color-pa-code second" style="background-color: <?php echo esc_attr($color['color_code']); ?>;"></div>
                        <div class="content-radio-color-pa-code third" style="background-color: <?php echo esc_attr($color['color_code']); ?>;"></div>
                        <div class="content-radio-color-pa-code fourth" style="background-color: <?php echo esc_attr($color['color_code']); ?>;"></div>
                        <span class="color-name"><?php echo esc_html($color['name']); ?></span>
                    </div>
                </div>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="check-icon">
                    <circle cx="12" cy="12" r="10" stroke="white" stroke-width="1.5" />
                    <path d="M8.5 12.5L10.5 14.5L15.5 9.5" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round" />
                </svg>
            </div>
        <?php
        }
    }

    public function widget($args, $instance)
    {
        // دریافت رنگ‌های انتخاب شده از URL
        $selected_colors = $this->get_selected_colors_from_url();
        $all_colors = $this->get_product_colors();

        // اگر رنگی موجود نیست، ویجت رو نمایش نده
        if (empty($all_colors)) {
            return;
        }

        // ایجاد لیست رنگ‌ها برای JavaScript
        $colors_data = array();
        foreach ($all_colors as $color) {
            $colors_data[$color['slug']] = $color['name'];
        }

        // بررسی آیا رنگی انتخاب شده یا نه
        $has_selected_colors = !empty($selected_colors);

        echo $args['before_widget'];
        ?>
        <section class="filter-product-color">
            <div class="filter-product-color-buttom <?php echo $has_selected_colors ? 'active' : ''; ?>">
                <span><?php echo __("رنگ", "lamkadeh"); ?></span>
                <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21.0813 15.96C20.8597 15.96 20.638 16.0417 20.463 16.2167L13.9997 22.68L7.53632 16.2167C7.19798 15.8783 6.63798 15.8783 6.29965 16.2167C5.96132 16.555 5.96132 17.115 6.29965 17.4533L13.3813 24.535C13.7197 24.8733 14.2796 24.8733 14.618 24.535L21.6997 17.4533C22.038 17.115 22.038 16.555 21.6997 16.2167C21.5363 16.0417 21.303 15.96 21.0813 15.96Z" fill="white" />
                    <path d="M14 3.20833C13.5217 3.20833 13.125 3.60499 13.125 4.08333L13.125 23.7183C13.125 24.1967 13.5217 24.5933 14 24.5933C14.4783 24.5933 14.875 24.1967 14.875 23.7183L14.875 4.08333C14.875 3.60499 14.4783 3.20833 14 3.20833Z" fill="white" />
                </svg>
                <div class="content">
                    <!-- بخش رنگ‌های انتخاب شده -->
                    <div class="content-color-me" id="selected-colors-container" <?php echo empty($selected_colors) ? 'style="display:none;"' : ''; ?>>
                        <span><?php echo __("انتخــاب شمـــــــــا", "lamkadeh") ?></span>
                        <div class="content-color-me-items" id="selected-colors-items">
                            <?php foreach ($selected_colors as $color_slug):
                                if (isset($colors_data[$color_slug])):
                                    $color_name = $colors_data[$color_slug];
                            ?>
                                    <div class="content-color-me-item" data-color-slug="<?php echo esc_attr($color_slug); ?>">
                                        <span><?php echo esc_html($color_name); ?></span>
                                        <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                        </svg>
                                    </div>
                            <?php
                                endif;
                            endforeach;
                            ?>
                        </div>
                    </div>
                    <span><?php echo __("رنگ", "lamkadeh") ?></span>
                    <div class="content-color">
                        <?php $this->render_colors_list($all_colors, $selected_colors); ?>
                    </div>
                </div>
            </div>
        </section>

        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const filterWidget = {
                    init() {
                        // مقداردهی اولیه از URL
                        this.selectedColors = this.getSelectedColorsFromURL();
                        this.colorsData = <?php echo json_encode($colors_data); ?>;

                        // اگر رنگی انتخاب شده، آکوردین رو active کن
                        if (this.selectedColors.length > 0) {
                            this.openAccordion();
                        }

                        this.bindEvents();
                        this.updateSelectedColorsUI();
                    },

                    // تابع باز کردن آکوردین
                    openAccordion() {
                        const accordionButton = document.querySelector('.filter-product-color-buttom');
                        accordionButton.classList.add('active');
                    },

                    // تابع بستن آکوردین
                    closeAccordion() {
                        const accordionButton = document.querySelector('.filter-product-color-buttom');
                        accordionButton.classList.remove('active');
                    },

                    // تابع دریافت رنگ‌های انتخاب شده از URL
                    getSelectedColorsFromURL() {
                        const urlParams = new URLSearchParams(window.location.search);
                        const colorParam = urlParams.get('product_color'); // استفاده از product_color
                        if (colorParam) {
                            return colorParam.split(',').filter(slug => slug.trim() !== '');
                        }
                        return [];
                    },

                    bindEvents() {
                        // آکاردیون - فقط روی دکمه اصلی کار کنه
                        const accordionButton = document.querySelector('.filter-product-color-buttom');

                        accordionButton.addEventListener('click', (e) => {
                            // فقط اگر روی خود دکمه اصلی کلیک شده (نه روی محتوای داخلش)
                            if (e.target === accordionButton ||
                                e.target.tagName.toLowerCase() === 'span' ||
                                e.target.tagName.toLowerCase() === 'svg' ||
                                e.target.tagName.toLowerCase() === 'path') {

                                // اگر روی content کلیک شده، کاری نکن
                                const content = accordionButton.querySelector('.content');
                                if (content.contains(e.target)) {
                                    return;
                                }

                                accordionButton.classList.toggle('active');
                            }
                        });

                        // کلیک روی رنگ
                        document.addEventListener('click', (e) => {
                            const colorItem = e.target.closest('.content-color-item');
                            if (colorItem) {
                                this.toggleColor(colorItem);
                            }

                            // حذف رنگ
                            const removeBtn = e.target.closest('.content-color-me-item');
                            if (removeBtn) {
                                this.removeColor(removeBtn.dataset.colorSlug);
                            }
                        });
                    },

                    toggleColor(colorItem) {
                        const colorSlug = colorItem.dataset.colorSlug;

                        if (this.selectedColors.includes(colorSlug)) {
                            // حذف رنگ
                            this.selectedColors = this.selectedColors.filter(slug => slug !== colorSlug);
                            colorItem.classList.remove('active');
                        } else {
                            // اضافه کردن رنگ
                            this.selectedColors.push(colorSlug);
                            colorItem.classList.add('active');
                        }

                        this.updateSelectedColorsUI();

                        // اگر رنگی انتخاب شده، آکوردین رو باز کن
                        if (this.selectedColors.length > 0) {
                            this.openAccordion();
                        }

                        this.updateURL();
                    },

                    removeColor(colorSlug) {
                        this.selectedColors = this.selectedColors.filter(slug => slug !== colorSlug);

                        // به‌روزرسانی وضعیت آیتم در لیست
                        const colorItem = document.querySelector(`.content-color-item[data-color-slug="${colorSlug}"]`);
                        if (colorItem) {
                            colorItem.classList.remove('active');
                        }

                        this.updateSelectedColorsUI();

                        // اگر هیچ رنگی انتخاب نشده، آکوردین رو ببند
                        if (this.selectedColors.length === 0) {
                            this.closeAccordion();
                        }

                        this.updateURL();
                    },

                    updateSelectedColorsUI() {
                        const container = document.getElementById('selected-colors-container');
                        const itemsContainer = document.getElementById('selected-colors-items');

                        itemsContainer.innerHTML = '';

                        // فقط رنگ‌هایی رو نمایش بده که در colorsData موجود هستن
                        const validSelectedColors = this.selectedColors.filter(slug => this.colorsData[slug]);

                        if (validSelectedColors.length === 0) {
                            container.style.display = 'none';
                            return;
                        }

                        container.style.display = 'block';

                        validSelectedColors.forEach(slug => {
                            const colorName = this.colorsData[slug];

                            if (colorName) {
                                const item = document.createElement('div');
                                item.className = 'content-color-me-item';
                                item.dataset.colorSlug = slug;
                                item.innerHTML = `
                                <span>${colorName}</span>
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M8.35891 7.4747C8.11483 7.23062 7.7191 7.23062 7.47502 7.4747C7.23095 7.71878 7.23095 8.11451 7.47502 8.35858L9.11642 9.99998L7.47504 11.6414C7.23096 11.8854 7.23096 12.2812 7.47504 12.5253C7.71912 12.7693 8.11485 12.7693 8.35893 12.5253L10.0003 10.8839L11.6417 12.5252C11.8858 12.7693 12.2815 12.7693 12.5256 12.5252C12.7696 12.2812 12.7696 11.8854 12.5256 11.6414L10.8842 9.99998L12.5256 8.3586C12.7697 8.11452 12.7697 7.7188 12.5256 7.47472C12.2815 7.23064 11.8858 7.23064 11.6417 7.47472L10.0003 9.1161L8.35891 7.4747Z" fill="#1D2977" />
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M10.0003 1.04166C5.05277 1.04166 1.04199 5.05244 1.04199 9.99999C1.04199 14.9475 5.05277 18.9583 10.0003 18.9583C14.9479 18.9583 18.9587 14.9475 18.9587 9.99999C18.9587 5.05244 14.9479 1.04166 10.0003 1.04166ZM2.29199 9.99999C2.29199 5.74279 5.74313 2.29166 10.0003 2.29166C14.2575 2.29166 17.7087 5.74279 17.7087 9.99999C17.7087 14.2572 14.2575 17.7083 10.0003 17.7083C5.74313 17.7083 2.29199 14.2572 2.29199 9.99999Z" fill="#1D2977" />
                                </svg>
                            `;
                                itemsContainer.appendChild(item);
                            }
                        });
                    },

                    updateURL() {
                        const url = new URL(window.location.href);

                        // فقط رنگ‌های معتبر رو نگه دار
                        const validSelectedColors = this.selectedColors.filter(slug => this.colorsData[slug]);

                        if (validSelectedColors.length > 0) {
                            url.searchParams.set('product_color', validSelectedColors.join(',')); // استفاده از product_color
                        } else {
                            url.searchParams.delete('product_color'); // استفاده از product_color
                        }

                        // رفرش صفحه با URL جدید
                        window.location.href = url.toString();
                    }
                };

                filterWidget.init();
            });
        </script>
    <?php
        echo $args['after_widget'];
    }

    public function form($instance)
    {
    ?>
        <p><?php echo __('این ویجت فیلتر رنگ محصولات را نمایش می‌دهد.', 'lamkadeh'); ?></p>
<?php
    }

    public function update($new_instance, $old_instance)
    {
        return $new_instance;
    }
}

add_action('widgets_init', function () {
    register_widget('Lamkadeh_Filter_Product_Color_Widget');
});

// اضافه کردن فیلتر برای صفحات آرشیو محصولات
add_filter('woocommerce_shortcode_products_query', 'lamkadeh_filter_shortcode_products_by_color', 10, 2);
function lamkadeh_filter_shortcode_products_by_color($query_args, $attributes)
{
    $selected_colors = array();

    // استفاده از پارامتر product_color به جای color
    if (isset($_GET['product_color'])) {
        if (is_array($_GET['product_color'])) {
            $selected_colors = array_map('sanitize_text_field', $_GET['product_color']);
        } else {
            $selected_colors = array_map('trim', explode(',', sanitize_text_field($_GET['product_color'])));
        }
    }

    $selected_colors = array_filter($selected_colors);

    if (!empty($selected_colors)) {
        if (!isset($query_args['tax_query'])) {
            $query_args['tax_query'] = array();
        }

        $query_args['tax_query'][] = array(
            'taxonomy' => 'pa_color',
            'field'    => 'slug',
            'terms'    => $selected_colors,
            'operator' => 'IN'
        );
    }

    return $query_args;
}
