<?php

require_once(get_template_directory() . "/senders/api-sms.php");

add_action('template_redirect', function () {
    if (function_exists('is_account_page') && is_account_page() && !is_user_logged_in()) {
        wc_get_template('/myaccount/form-login.php');
        exit;
    }
});

function lamkadeh_save_otp($phone, $type, $otp, $name = '')
{
    $data = array(
        'otp' => $otp,
        'name' => $name,
        'attempts' => 0,
        'created_at' => time(),
        'expires_at' => time() + (10 * 60) // 10 دقیقه
    );

    $key = 'lamkadeh_otp_' . $phone . '_' . $type;
    $saved = update_option($key, $data, false);

    // دیباگ
    error_log("📱 LAMKADEH OTP SAVED");
    error_log("Key: " . $key);
    error_log("OTP: " . $otp);
    error_log("Saved: " . ($saved ? '✅ YES' : '❌ NO'));

    return $saved;
}

function lamkadeh_get_otp($phone, $type)
{
    $key = 'lamkadeh_otp_' . $phone . '_' . $type;
    $data = get_option($key);

    if (!$data) {
        error_log("❌ LAMKADEH OTP NOT FOUND: " . $key);
        return false;
    }

    // بررسی انقضا
    if (time() > $data['expires_at']) {
        delete_option($key);
        error_log("❌ LAMKADEH OTP EXPIRED: " . $key);
        return false;
    }

    error_log("✅ LAMKADEH OTP FOUND: " . $data['otp']);
    return $data;
}

function lamkadeh_delete_otp($phone, $type)
{
    $key = 'lamkadeh_otp_' . $phone . '_' . $type;
    return delete_option($key);
}

function lamkadeh_increment_attempts($phone, $type)
{
    $data = lamkadeh_get_otp($phone, $type);
    if ($data) {
        $data['attempts']++;
        $key = 'lamkadeh_otp_' . $phone . '_' . $type;
        update_option($key, $data, false);
        return $data['attempts'];
    }
    return 0;
}

// نرمال سازی شماره تلفن
function lamkadeh_normalize_phone($phone)
{
    $phone = preg_replace('/[^0-9]/', '', $phone);
    if (preg_match('/^9[0-9]{9}$/', $phone)) {
        $phone = '0' . $phone;
    }
    return $phone;
}

// ==================== Ajax Handlers ====================

// ارسال کد OTP
add_action('wp_ajax_nopriv_lamkadeh_send_otp', 'lamkadeh_handle_send_otp');
add_action('wp_ajax_lamkadeh_send_otp', 'lamkadeh_handle_send_otp');
function lamkadeh_handle_send_otp()
{
    try {
        // بررسی nonce
        if (!wp_verify_nonce($_POST['nonce'], 'lamkadeh_otp_nonce')) {
            throw new Exception('درخواست نامعتبر است.');
        }

        // دریافت و اعتبارسنجی داده‌ها
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $type = sanitize_text_field($_POST['type'] ?? '');
        $name = sanitize_text_field($_POST['name'] ?? '');

        if (empty($phone)) {
            throw new Exception('شماره موبایل الزامی است.');
        }

        // نرمال سازی شماره
        $phone = lamkadeh_normalize_phone($phone);

        if (!preg_match('/^09[0-9]{9}$/', $phone)) {
            throw new Exception('شماره موبایل معتبر نیست.');
        }

        // بررسی نوع درخواست
        if ($type === 'register') {
            $existing_user = get_user_by('login', $phone);
            if ($existing_user) {
                throw new Exception('این شماره موبایل قبلاً ثبت‌نام کرده است.');
            }
        } elseif ($type === 'login') {
            $user = get_user_by('login', $phone);
            if (!$user) {
                throw new Exception('کاربری با این شماره موبایل وجود ندارد.');
            }
        } else {
            throw new Exception('نوع درخواست نامعتبر است.');
        }

        // تولید و ذخیره OTP
        $otp = generateCodeOtp();
        $saved = lamkadeh_save_otp($phone, $type, $otp, $name);

        if (!$saved) {
            throw new Exception('خطا در ذخیره سازی کد.');
        }

        // ارسال پیامک
        $sms_result = sendSms($phone, $otp);

        if ($sms_result) {
            wp_send_json_success('کد تایید به شماره ' . $phone . ' ارسال شد.');
        } else {
            throw new Exception('ارسال پیامک ناموفق بود.');
        }
    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
}

// تأیید کد OTP
add_action('wp_ajax_nopriv_lamkadeh_verify_otp', 'lamkadeh_handle_verify_otp');
add_action('wp_ajax_lamkadeh_verify_otp', 'lamkadeh_handle_verify_otp');
function lamkadeh_handle_verify_otp()
{
    try {
        // بررسی nonce
        if (!wp_verify_nonce($_POST['nonce'], 'lamkadeh_otp_nonce')) {
            throw new Exception('درخواست نامعتبر است.');
        }

        // دریافت و اعتبارسنجی داده‌ها
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $code = sanitize_text_field($_POST['code'] ?? '');
        $type = sanitize_text_field($_POST['type'] ?? '');
        $name = sanitize_text_field($_POST['name'] ?? '');

        // دیباگ کامل
        error_log("=== LAMKADEH VERIFY DEBUG ===");
        error_log("Phone: " . $phone);
        error_log("Code: " . $code);
        error_log("Type: " . $type);
        error_log("Name: " . $name);

        if (empty($phone) || empty($code)) {
            throw new Exception('شماره موبایل و کد تایید الزامی است.');
        }


        // نرمال سازی شماره
        $phone = lamkadeh_normalize_phone($phone);

        // دریافت OTP
        $otp_data = lamkadeh_get_otp($phone, $type);
        if (!$otp_data) {
            throw new Exception('کد منقضی شده است. لطفاً مجدداً درخواست کد دهید.');
        }

        // بررسی تعداد تلاش‌ها
        $attempts = lamkadeh_increment_attempts($phone, $type);
        if ($attempts >= 3) {
            lamkadeh_delete_otp($phone, $type);
            throw new Exception('تعداد تلاش‌های شما بیش از حد مجاز است. لطفاً مجدداً درخواست کد دهید.');
        }

        // بررسی صحت کد
        if (strval($otp_data['otp']) !== strval($code)) {
            $remaining = 3 - $attempts;
            throw new Exception('کد وارد شده صحیح نیست. ' . $remaining . ' تلاش باقی مانده است.');
        }

        // پردازش موفق
        if ($type === 'login') {
            lamkadeh_handle_login($phone);
        } elseif ($type === 'register') {
            lamkadeh_handle_register($phone, $name);
        }

        // پاکسازی
        lamkadeh_delete_otp($phone, $type);

        wp_send_json_success(array(
            'message' => $type === 'login' ? 'ورود موفقیت‌آمیز بود.' : 'ثبت‌نام و ورود موفقیت‌آمیز بود.',
            'redirect' => wc_get_page_permalink('myaccount')
        ));
    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
}

// پردازش ورود
function lamkadeh_handle_login($phone)
{
    $user = get_user_by('login', $phone);
    if (!$user) {
        throw new Exception('کاربری با این شماره موبایل وجود ندارد.');
    }

    wp_clear_auth_cookie();
    wp_set_current_user($user->ID);
    wp_set_auth_cookie($user->ID, true);
    update_user_meta($user->ID, 'last_login', current_time('mysql'));
}

// پردازش ثبت‌نام
function lamkadeh_handle_register($phone, $name)
{
    // بررسی وجود کاربر
    $existing_user = get_user_by('login', $phone);
    if ($existing_user) {
        throw new Exception('این شماره موبایل قبلاً ثبت‌نام کرده است.');
    }

    // ایجاد کاربر
    $password = wp_generate_password(12, true, true);
    $user_id = wp_create_user($phone, $password, $phone);

    if (is_wp_error($user_id)) {
        throw new Exception('خطا در ایجاد حساب کاربری: ' . $user_id->get_error_message());
    }

    // به روزرسانی اطلاعات
    if (!empty($name)) {
        $name_parts = explode(' ', $name, 2);
        $first_name = $name_parts[0];
        $last_name = isset($name_parts[1]) ? $name_parts[1] : '';

        wp_update_user([
            'ID' => $user_id,
            'display_name' => $name,
            'first_name' => $first_name,
            'last_name' => $last_name
        ]);
    }

    // ذخیره متادیتا
    update_user_meta($user_id, 'billing_phone', $phone);
    update_user_meta($user_id, 'phone', $phone);
    update_user_meta($user_id, 'registration_date', current_time('mysql'));

    // ورود خودکار
    wp_clear_auth_cookie();
    wp_set_current_user($user_id);
    wp_set_auth_cookie($user_id, true);
}

// ==================== ابزارهای دیباگ ====================

// صفحه دیباگ
add_action('wp_ajax_nopriv_lamkadeh_debug', 'lamkadeh_debug_page');
add_action('wp_ajax_lamkadeh_debug', 'lamkadeh_debug_page');
function lamkadeh_debug_page()
{
    global $wpdb;

    echo '<h1>لمکده - دیباگ سیستم OTP</h1>';

    // نمایش تمام OTPهای ذخیره شده
    $results = $wpdb->get_results(
        "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE 'lamkadeh_otp_%'"
    );

    echo '<h2>OTPهای ذخیره شده:</h2>';
    if (empty($results)) {
        echo '<p>هیچ OTPی ذخیره شده‌ای وجود ندارد.</p>';
    } else {
        foreach ($results as $row) {
            $value = maybe_unserialize($row->option_value);
            echo '<div style="border:1px solid #ccc; padding:10px; margin:10px;">';
            echo '<strong>' . $row->option_name . '</strong><br>';
            echo '<pre>' . print_r($value, true) . '</pre>';
            echo '</div>';
        }
    }

    wp_die();
}

// تست ساده
add_action('wp_ajax_nopriv_lamkadeh_test', 'lamkadeh_test');
add_action('wp_ajax_lamkadeh_test', 'lamkadeh_test');
function lamkadeh_test()
{
    wp_send_json_success('سرور در دسترس است!');
}

add_action('show_user_profile', 'lamkadeh_add_user_avatar_field');
add_action('edit_user_profile', 'lamkadeh_add_user_avatar_field');
function lamkadeh_add_user_avatar_field($user)
{
    $avatar = get_user_meta($user->ID, 'lamkadeh_user_avatar', true);
    ?>
    <h3><?php echo __("تصویر پروفایل", "lamkadeh") ?></h3>
    <table class="form-table">
        <tr>
            <th><label for="lamkadeh_user_avatar"><?php echo __("آپلود تصویر جدید", "lamkadeh") ?></label></th>
            <td>
                <?php if ($avatar): ?>
                    <img src="<?php echo esc_url($avatar); ?>" width="100" height="100"
                        style="border-radius:50%;object-fit:cover;"><br>
                <?php endif; ?>
                <input type="file" name="lamkadeh_user_avatar" id="lamkadeh_user_avatar"><br>
                <span class="description"><?php echo __("یک تصویر برای پروفایل خود آپلود کنید", "lamkadeh") ?></span>
            </td>
        </tr>
    </table>
    <?php
}

function lamkadeh_save_user_avatar($user_id)
{
    if (!current_user_can('edit_user', $user_id)) {
        return false;
    }

    if (!empty($_FILES['lamkadeh_user_avatar']['name'])) {

        $file = $_FILES['lamkadeh_user_avatar'];
        $allowed = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        if (!in_array($ext, $allowed)) {
            return;
        }

        require_once(ABSPATH . 'wp-admin/includes/file.php');
        $upload = wp_handle_upload($file, ['test_form' => false]);

        if (isset($upload['url']) && !isset($upload['error'])) {
            update_user_meta($user_id, 'lamkadeh_user_avatar', esc_url($upload['url']));
        } else {
            error_log('Avatar upload failed: ' . print_r($upload, true));
        }
    }
}
add_action('personal_options_update', 'lamkadeh_save_user_avatar');
add_action('edit_user_profile_update', 'lamkadeh_save_user_avatar');

function lamkadeh_force_user_avatar($avatar, $id_or_email, $size, $default, $alt)
{
    $user = false;

    if (is_numeric($id_or_email)) {
        $user = get_user_by('id', $id_or_email);
    } elseif (is_object($id_or_email)) {
        if (!empty($id_or_email->user_id)) {
            $user = get_user_by('id', $id_or_email->user_id);
        } elseif (!empty($id_or_email->comment_author_email)) {
            $user = get_user_by('email', $id_or_email->comment_author_email);
        }
    } elseif (is_string($id_or_email) && is_email($id_or_email)) {
        $user = get_user_by('email', $id_or_email);
    }

    if ($user) {
        $custom_avatar = get_user_meta($user->ID, 'lamkadeh_user_avatar', true);
        if (!empty($custom_avatar)) {
            $avatar = sprintf(
                '<img alt="%s" src="%s" class="avatar avatar-%d photo" height="%d" width="%d" style="border-radius:50%%;object-fit:cover;">',
                esc_attr($alt),
                esc_url($custom_avatar),
                (int) $size,
                (int) $size,
                (int) $size
            );
        }
    }

    if (empty($avatar)) {
        $fallback = get_template_directory_uri() . '/assets/images/theme/avatar.png';
        $avatar = sprintf(
            '<img alt="%s" src="%s" class="avatar avatar-%d photo" height="%d" width="%d" style="border-radius:50%%;object-fit:cover;">',
            esc_attr($alt),
            esc_url($fallback),
            (int) $size,
            (int) $size,
            (int) $size
        );
    }

    return $avatar;
}
add_filter('get_avatar', 'lamkadeh_force_user_avatar', 9999, 5);

add_action('user_edit_form_tag', function () {
    echo ' enctype="multipart/form-data"';
});

function lamakdeh_user_multiple_addresses($user)
{
    $addresses = get_user_meta($user->ID, 'additional_addresses', true);
    if (!is_array($addresses))
        $addresses = [];
    ?>
    <h3>آدرس‌های اضافی</h3>
    <table class="form-table">
        <tr>
            <th><label for="additional_addresses">آدرس‌ها</label></th>
            <td>
                <textarea name="additional_addresses" rows="5"
                    style="width:100%;"><?php echo esc_textarea(json_encode($addresses, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)); ?></textarea>
                <p class="description">آدرس‌ها را به صورت JSON ذخیره کن. مثال:<br>
                    [{"address":"تهران - خیابان ولیعصر","city":"تهران","postcode":"12345"}]</p>
            </td>
        </tr>
    </table>
    <?php
}
add_action('show_user_profile', 'lamakdeh_user_multiple_addresses');
add_action('edit_user_profile', 'lamakdeh_user_multiple_addresses');

function lamakdeh_save_user_multiple_addresses($user_id)
{
    if (isset($_POST['additional_addresses'])) {
        $json = stripslashes($_POST['additional_addresses']);
        $data = json_decode($json, true);
        if (is_array($data)) {
            update_user_meta($user_id, 'additional_addresses', $data);
        }
    }
}
add_action('personal_options_update', 'lamakdeh_save_user_multiple_addresses');
add_action('edit_user_profile_update', 'lamakdeh_save_user_multiple_addresses');

add_action('woocommerce_my_account_my_orders_query', function ($query) {
    if (isset($_GET['status']) && $_GET['status'] !== 'all') {
        $status = sanitize_text_field($_GET['status']);
        $query['status'] = $status;
    }
    return $query;
});

add_action('woocommerce_save_account_details', function ($user_id) {
    if (!empty($_FILES['profile_image_upload']['name'])) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';

        $uploaded = media_handle_upload('profile_image_upload', 0);
        if (!is_wp_error($uploaded)) {
            update_user_meta($user_id, 'lamkadeh_user_avatar', wp_get_attachment_url($uploaded));
        }
    }
});

add_action('woocommerce_edit_account_form', 'lamakdeh_show_readonly_phone_field');
function lamakdeh_show_readonly_phone_field()
{
    $user_id = get_current_user_id();
    $billing_phone = get_user_meta($user_id, 'billing_phone', true);

    if ($billing_phone) {
        ?>
        <p class="woocommerce-form-row woocommerce-form-row--first form-row form-row-wide">
            <label for="billing_phone"><?php _e('شماره تلفن همراه', 'lamakdeh'); ?></label>
            <input type="text" readonly disabled name="billing_phone_display" id="billing_phone_display"
                value="<?php echo esc_attr($billing_phone); ?>" class="input-text" />
        </p>
        <?php
    }
}

add_action('woocommerce_save_account_details', 'lamakdeh_prevent_phone_edit', 10, 1);
function lamakdeh_prevent_phone_edit($user_id)
{
    if (isset($_POST['billing_phone_display'])) {
        unset($_POST['billing_phone']);
    }
}

add_filter('woocommerce_account_orders_columns', 'lamkadeh_custom_account_order_columns');
function lamkadeh_custom_account_order_columns($columns)
{
    if (isset($columns['order-date'])) {
        $columns['order-date'] = __('تاریخ سفارش', 'lamkadeh');
    }

    if (isset($columns['order-number'])) {
        $columns['order-number'] = __('شماره سفارش', 'lamkadeh');
    }

    if (isset($columns['order-actions'])) {
        $columns['order-actions'] = __('جزئیات سفارش', 'lamkadeh');
    }

    if (isset($columns['order-total'])) {
        $columns['order-total'] = __('مبلغ سفارش', 'lamkadeh');
    }

    if (isset($columns['order-status'])) {
        $columns['order-status'] = __('وضعیت سفارش', 'lamkadeh');
    }

    $new = [];
    foreach ($columns as $key => $label) {

        $new[$key] = $label;

        if ($key === 'order-number') {
            $new['items-count'] = __('تعداد اقلام', 'lamkadeh');
        }
    }

    return $new;
}

add_action('woocommerce_my_account_my_orders_column_order-date', 'lamkadeh_show_custom_order_date');
function lamkadeh_show_custom_order_date($order)
{

    require_once(get_template_directory() . "/includes/date-convertor-shamsi/jdf.php");

    $timestamp = $order->get_date_created()->getTimestamp();

    if (function_exists('jdate')) {
        echo esc_html(jdate('Y/m/d', $timestamp));
    } else {
        echo esc_html(date('Y/m/d', $timestamp));
    }
}

add_action('woocommerce_my_account_my_orders_column_items-count', 'lamkadeh_show_items_count_column');
function lamkadeh_show_items_count_column($order)
{

    $count = 0;
    foreach ($order->get_items() as $item) {
        $count += $item->get_quantity();
    }

    echo esc_html($count . ' ' . __('کالا', 'lamkadeh'));
}

/**
 * Place order-status column before order-actions in My Account orders table
 */
add_filter( 'woocommerce_account_orders_columns', 'lamkadeh_move_order_status_column', 20 );
function lamkadeh_move_order_status_column( $columns ) {
    if ( ! isset( $columns['order-status'] ) ) {
        return $columns;
    }
    $status_col = $columns['order-status'];
    unset( $columns['order-status'] );
    $new_columns = [];

    $insert_before = 'order-actions';

    $inserted = false;
    foreach ( $columns as $key => $label ) {

        if ( $key === $insert_before ) {
            $new_columns['order-status'] = $status_col;
            $inserted = true;
        }
        $new_columns[ $key ] = $label;
    }
    if ( ! $inserted ) {
        $new_columns['order-status'] = $status_col;
    }

    return $new_columns;
}